/*
 Copyright (c) 2007, Distributed Computing Group (DCG)
                    ETH Zurich
                    Switzerland
                    dcg.ethz.ch

 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

 - Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the
   distribution.

 - Neither the name 'Sinalgo' nor the names of its contributors may be
   used to endorse or promote products derived from this software
   without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
package sinalgo.models;

import sinalgo.nodes.Position;

/**
 * A distribution model describes how the nodes are (initially) palced in the simulation area.
 * <p>
 * The framework will ask the positoin of nodes in sequence. The method <code>getNextPosition()</code>
 * allows the framework to iterate over all nodes and assign them a position. 
 * <p>
 * The field numberOfNodes is always set before geNextPosition is called the first time. 
 */
public abstract class DistributionModel extends Model {

	/**
	 * The number of nodes, that have to be generated by this model.
	 */
	protected int numberOfNodes = -1;
	
	
	/**
	 * Method that gets galled after the instance has been created and the number of nodes has been set. It
	 * is used to set all the parameters that need to know the number of nodes that ar created.
	 */
	public void initialize(){
		// empty body - overwrite in the subclass, if needed.
	}
	
	/**
	 * Sets the number of nodes this distribution model will have to generate. 
	 * <p>
	 * This information must be given before the first call to <code>getOnePosition()</code>. 
	 * @param num
	 */
	public void setNumberOfNodes(int num) {
		numberOfNodes = num;
	}
	
	/**
	 * Returns the next position where a node is placed.
	 * <p>
	 * You may precalculate all positions and store them in a datastructure. Then, return one after the other
	 * of these positions when this method is called.  
	 * @return The next position where a node is placed.
	 */
	public abstract Position getNextPosition(); 

	/* (non-Javadoc)
	 * @see models.Model#getType()
	 */
	public final ModelType getType() {
		return ModelType.DistributionModel;
	}
}
